# =====================================================================
# Programme : Chrono Pico v2.3.0 (bi-cœur)
# Auteur    : François / framboise314
# Licence   : MIT
# Date      : 2025-11-02
# Matériel  : Raspberry Pi Pico + SSD1306 128x64 (I2C addr 0x3C)
# ---------------------------------------------------------------------
# Invariants d'UI & fonctionnement (identiques v2.2.1) :
#  - Démarrage normal : bandeau "CHRONO" + ▶/■ + chrono 7-seg (v1.0.4)
#    + batterie v1.6.2 (barres + éclair GP13, message "RECHARGER / LA BATTERIE")
#  - Démarrage RAZ maintenu (mode DIAG) :
#      * A GAUCHE : température interne (xx.x°C) à la place de "CHRONO"
#      * EN HAUT-DROITE : batterie avec la tension VSYS au centre du corps
#      * Eclair si charge (GP13=LOW)
#  - I2C : GP0/GP1 @ 400 kHz, addr 0x3C. Pas d’auto-détection.
#  - VSYS : ADC29 (VSYS/3) → ×3. USB présent si VSYS ≥ 4.5 V.
# ---------------------------------------------------------------------
# Nouvelle archi v2.3.0 :
#  - Cœur 1 (thread) = LOGIQUE CHRONO + LECTURE BOUTONS (A/M, RAZ)
#  - Cœur 0 (main)   = AFFICHAGE OLED + ADC VSYS + TEMP + BUZZER
#  - Communication via état partagé protégé par mutex.
# =====================================================================

from machine import Pin, I2C, ADC
from ssd1306 import SSD1306_I2C
import time, _thread
from logo_bitmap import draw_logo   # logo centré, fond blanc (v1.6.2)

# ------------------------- I2C / OLED -------------------------------
I2C_SDA_PIN = 0
I2C_SCL_PIN = 1
I2C_FREQ    = 400_000
OLED_W, OLED_H = 128, 64
I2C_ADDR    = 0x3C

# ------------------------- MARGES / TEXTE CHRONO --------------------
MARGE_ECRAN = 6
TEXTE_CHRONO       = "CHRONO"
CHRONO_X           = 6
CHRONO_Y           = 9
CHRONO_ECHELLE     = 1
CHRONO_ESPACE_PX   = 1
SYMBOLE_ESPACE_APRES_TEXTE = 5

# ------------------------- CHRONO (5 BOÎTES) ------------------------
X_BOITE0          = MARGE_ECRAN
Y_BOITE0          = OLED_H - MARGE_ECRAN - 30
LARGEUR_BOITE     = 21
HAUTEUR_BOITE     = 30
ECART_ENTRE_BOITES = 2
ECART_AVANT_POINTS = 2
ECART_APRES_POINTS = 2
LARGEUR_POINTS     = 2
ECART_VERTICAL_POINTS = 8
ECART_INTERNE_DANS_BOITE5 = 2
LARGEUR_BOITE5 = 2 * LARGEUR_BOITE + ECART_INTERNE_DANS_BOITE5

# ------------------------- CHIFFRES 7-seg ---------------------------
SEG_EPAISSEUR = 3
SEG_MARGE     = 2
MAP_7SEG = {
    '0': 0b1111110, '1': 0b0110000, '2': 0b1101101, '3': 0b1111001,
    '4': 0b0110011, '5': 0b1011011, '6': 0b1011111, '7': 0b1110000,
    '8': 0b1111111, '9': 0b1111011,
}

# ------------------------- BOUTONS (polling sur cœur 1) -------------
BP_AM_PIN  = 14  # Start/Stop (A/M)
BP_RAZ_PIN = 15  # RAZ (seulement à l'arrêt)
DEBOUNCE_MS = 20

# ------------------------- BUZZER (cœur 0) --------------------------
BUZZER_PIN = 28
BEEP_MS    = 1
buzzer = Pin(BUZZER_PIN, Pin.OUT); buzzer.value(0)
_beep_until = None
def beep_now(now_ms=None, dur_ms=BEEP_MS):
    global _beep_until
    now = time.ticks_ms() if now_ms is None else now_ms
    buzzer.value(1); _beep_until = time.ticks_add(now, dur_ms)
def _service_beep(now_ms=None):
    global _beep_until
    if _beep_until is None: return
    now = time.ticks_ms() if now_ms is None else now_ms
    if time.ticks_diff(now, _beep_until) >= 0:
        buzzer.value(0); _beep_until = None

# ------------------------- TEMPOS (cœur 0) --------------------------
PERIODE_RAFRAICH_MS = 50
PERIODE_BATT_MS     = 1000
SPLASH_DUREE_MS     = 2000
BLINK_PERIODE_MS    = 500

# ------------------------- ADC VSYS/3 (cœur 0) ----------------------
ADC_VSYS_CH   = 29       # ADC3 (GPIO29)
FACTEUR_ADC   = 3.3 / 65535.0
FACTEUR_VSYS  = 3.0      # VSYS/3 → ×3
USB_VSYS_MIN  = 4.5
def init_adc_vsys(): return ADC(ADC_VSYS_CH)
def lire_vsys(adc_vsys):
    brut = adc_vsys.read_u16()
    return (brut * FACTEUR_ADC) * FACTEUR_VSYS

# ------------------------- Température interne (cœur 0) -------------
ADC_TEMP_CH   = 4
NB_ECH_TEMP   = 8
ADC_VREF      = 3.3
def init_adc_temp(): return ADC(ADC_TEMP_CH)
def lire_temp_c(adc_temp):
    total = 0
    for _ in range(NB_ECH_TEMP):
        total += adc_temp.read_u16()
    moy = total / NB_ECH_TEMP
    v = (moy / 65535.0) * ADC_VREF
    return 27.0 - (v - 0.706) / 0.001721  # °C (datasheet)

# ------------------------- Détection charge (cœur 0) ----------------
PIN_CHARGE_DET = 13  # ACTIF BAS (LED CHRG ON)
charge_det = Pin(PIN_CHARGE_DET, Pin.IN, Pin.PULL_UP)
def charge_en_cours(): return charge_det.value() == 0

# ========================= OUTILS DESSIN (cœur 0) ===================
def rect_plein(oled, x, y, w, h, c=1):
    for yy in range(y, y + h): oled.hline(x, yy, w, c)

FONT_5x7 = {
    'A':[0b01110,0b10001,0b10001,0b11111,0b10001,0b10001,0b10001],
    'B':[0b11110,0b10001,0b11110,0b10001,0b10001,0b10001,0b11110],
    'C':[0b01110,0b10001,0b10000,0b10000,0b10000,0b10001,0b01110],
    'D':[0b11100,0b10010,0b10001,0b10001,0b10001,0b10010,0b11100],
    'E':[0b11111,0b10000,0b11110,0b10000,0b10000,0b10000,0b11111],
    'G':[0b01110,0b10001,0b10000,0b10111,0b10001,0b10001,0b01110],
    'H':[0b10001,0b10001,0b10001,0b11111,0b10001,0b10001,0b10001],
    'I':[0b11111,0b00100,0b00100,0b00100,0b00100,0b00100,0b11111],
    'L':[0b10000,0b10000,0b10000,0b10000,0b10000,0b10000,0b11111],
    'N':[0b10001,0b11001,0b10101,0b10011,0b10001,0b10001,0b10001],
    'O':[0b01110,0b10001,0b10001,0b10001,0b10001,0b10001,0b01110],
    'R':[0b11110,0b10001,0b10001,0b11110,0b10100,0b10010,0b10001],
    'T':[0b11111,0b00100,0b00100,0b00100,0b00100,0b00100,0b00100],
    '0':[0b01110,0b10001,0b10011,0b10101,0b11001,0b10001,0b01110],
    '1':[0b00100,0b01100,0b00100,0b00100,0b00100,0b00100,0b01110],
    '2':[0b01110,0b10001,0b00001,0b00110,0b11000,0b10000,0b11111],
    '3':[0b11110,0b00001,0b00001,0b01110,0b00001,0b00001,0b11110],
    '4':[0b00010,0b00110,0b01010,0b10010,0b11111,0b00010,0b00010],
    '5':[0b11111,0b10000,0b11110,0b00001,0b00001,0b10001,0b01110],
    '6':[0b00110,0b01000,0b10000,0b11110,0b10001,0b10001,0b01110],
    '7':[0b11111,0b00001,0b00010,0b00100,0b01000,0b10000,0b10000],
    '8':[0b01110,0b10001,0b10001,0b01110,0b10001,0b10001,0b01110],
    '9':[0b01110,0b10001,0b10001,0b01111,0b00001,0b00010,0b01100],
    '.':[0b00000,0b00000,0b00000,0b00000,0b00000,0b01100,0b01100],
    '-':[0b00000,0b00000,0b00000,0b01110,0b00000,0b00000,0b00000],
    '°':[0b01110,0b10001,0b10001,0b01110,0b00000,0b00000,0b00000],
    'C2':[0b01110,0b10001,0b10000,0b10000,0b10000,0b10001,0b01110],
    'V':[0b10001,0b10001,0b10001,0b10001,0b01010,0b01010,0b00100],
    ' ': [0,0,0,0,0,0,0],
}
def dessiner_car_5x7(oled, x, y, car, echelle=1, couleur=1):
    key = car if car in FONT_5x7 else ('C2' if car=='C' else ' ')
    matrice = FONT_5x7[key]
    for lig, bits in enumerate(matrice):
        for col in range(5):
            if (bits >> (4 - col)) & 1:
                rect_plein(oled, x + col*echelle, y + lig*echelle, echelle, echelle, couleur)
def dessiner_texte_5x7(oled, x, y, texte, echelle=1, esp=1, couleur=1):
    cx = x; pas = 5*echelle + esp*echelle
    for ch in texte:
        dessiner_car_5x7(oled, cx, y, ch, echelle, couleur); cx += pas
    return cx
def mesurer_texte_5x7(texte, echelle=1, esp=1):
    if not texte: return 0
    n = len(texte); return n*(5*echelle) + (n-1)*(esp*echelle)
def dessiner_triangle_7(oled, x, y, c=1):
    for dy, w in enumerate([1,2,3,4,3,2,1]): rect_plein(oled, x, y+dy, w, 1, c)
def dessiner_carre_7(oled, x, y, c=1):
    rect_plein(oled, x, y, 7, 7, c)

# ------------------------- Batterie (v1.6.2) ------------------------
NB_BARRES         = 5
LARGEUR_BARRE     = 6
ECART_BARRES      = 2
MARGE_INTERNE_X   = 3
MARGE_INTERNE_Y   = 3
HAUTEUR_INTERNE   = 10
LARGEUR_INTERNE   = NB_BARRES * LARGEUR_BARRE + (NB_BARRES - 1) * ECART_BARRES
LARGEUR_CORPS     = LARGEUR_INTERNE + 2 * MARGE_INTERNE_X
HAUTEUR_CORPS     = HAUTEUR_INTERNE + 2 * MARGE_INTERNE_Y
X_BATT            = OLED_W - MARGE_ECRAN - LARGEUR_CORPS
Y_BATT            = MARGE_ECRAN
BORNE_LARGEUR     = 3
BORNE_HAUTEUR     = 6
ECLAIR_BITMAP = [
    "0000000011111",
    "0000000111110",
    "0000001111100",
    "0000011111000",
    "0000111110000",
    "0001111100000",
    "0000001111100",
    "0000011111000",
    "0000111100000",
    "0001110000000",
    "0011100000000",
    "0110000000000",
    "1000000000000",
]
ECLAIR_W = len(ECLAIR_BITMAP[0]); ECLAIR_H = len(ECLAIR_BITMAP)
ECLAIR_MARGE_GAUCHE = 2
def dessiner_contour_batterie(oled, x, y, w, h): oled.rect(x, y, w, h, 1)
def dessiner_borne_plus(oled, x_corps, y_corps, h_corps):
    x_b = x_corps - BORNE_LARGEUR - 1
    y_b = y_corps + h_corps // 2 - BORNE_HAUTEUR // 2
    rect_plein(oled, x_b, y_b, BORNE_LARGEUR, BORNE_HAUTEUR, 1)
def dessiner_barres(oled, x_corps, y_corps, niveau):
    x0 = x_corps + MARGE_INTERNE_X; y0 = y_corps + MARGE_INTERNE_Y
    for i in range(NB_BARRES):
        x_barre = x0 + i * (LARGEUR_BARRE + ECART_BARRES)
        on = (i >= NB_BARRES - niveau)
        rect_plein(oled, x_barre, y0, LARGEUR_BARRE, HAUTEUR_INTERNE, 1 if on else 0)
def dessiner_eclair_bitmap(oled, x_batt, y_batt):
    x_gauche_borne = x_batt - 1
    x_eclair = x_gauche_borne - BORNE_LARGEUR - ECLAIR_MARGE_GAUCHE - ECLAIR_W
    y_eclair = y_batt + (HAUTEUR_CORPS - ECLAIR_H) // 2
    for dy, ligne in enumerate(ECLAIR_BITMAP):
        yy = y_eclair + dy
        for dx, ch in enumerate(ligne):
            if ch == '1': oled.pixel(x_eclair + dx, yy, 1)
def dessiner_batterie(oled, x, y, niveau, eclair=False):
    dessiner_contour_batterie(oled, x, y, LARGEUR_CORPS, HAUTEUR_CORPS)
    dessiner_borne_plus(oled, x, y, HAUTEUR_CORPS)
    dessiner_barres(oled, x, y, niveau)
    if eclair: dessiner_eclair_bitmap(oled, x, y)

# ------------------------- Niveaux de barres ------------------------
def niveau_barres_pour_vsys(v):
    if v >= 4.20: return 5
    if v >= 4.00: return 4
    if v >= 3.85: return 3
    if v >= 3.75: return 2
    if v >= 3.60: return 1
    return 0

# ------------------------- Messages / DIAG --------------------------
def afficher_message_recharger(oled):
    texte1 = "RECHARGER"; texte2 = "LA BATTERIE"
    w1 = mesurer_texte_5x7(texte1, 1, 1)
    w2 = mesurer_texte_5x7(texte2, 1, 1)
    x1 = (OLED_W - w1) // 2; x2 = (OLED_W - w2) // 2
    y_base = Y_BOITE0 + 6
    dessiner_texte_5x7(oled, x1, y_base, texte1, 1, 1, 1)
    dessiner_texte_5x7(oled, x2, y_base + 10, texte2, 1, 1, 1)

def afficher_vsys_dans_batterie(oled, vsys_v, eclair):
    dessiner_batterie(oled, X_BATT, Y_BATT, 0, eclair=eclair)
    txt = "{:.2f}V".format(vsys_v)
    tw = mesurer_texte_5x7(txt, 1, 1)
    tx = X_BATT + (LARGEUR_CORPS - tw)//2
    ty = Y_BATT + (HAUTEUR_CORPS - 7)//2
    dessiner_texte_5x7(oled, tx, ty, txt, 1, 1, 1)

def afficher_temperature_gauche(oled, temp_c):
    txt = "{:.1f}°C".format(temp_c)
    dessiner_texte_5x7(oled, CHRONO_X, CHRONO_Y, txt, 1, 1, 1)

# ========================= Chrono 7-seg =============================
def seg_rects(x, y, w, h, ep, marge):
    xi = x + marge; yi = y + marge
    wi = max(1, w - 2*marge); hi = max(3, h - 2*marge)
    e  = max(1, min(ep, hi // 4))
    a = (xi + e, yi,                     wi - 2*e, e)
    g = (xi + e, yi + (hi//2) - e//2,    wi - 2*e, e)
    d = (xi + e, yi + hi - e,            wi - 2*e, e)
    b = (xi + wi - e, yi + e,            e, (hi//2) - e)
    c = (xi + wi - e, yi + (hi//2) + e//2, e, (hi//2) - e)
    f = (xi,          yi + e,            e, (hi//2) - e)
    eS= (xi,          yi + (hi//2) + e//2, e, (hi//2) - e)
    return [a,b,c,d,eS,f,g]
def dessiner_digit(oled, x, y, w, h, ch):
    code = MAP_7SEG.get(ch, 0)
    for i, (rx, ry, rw, rh) in enumerate(seg_rects(x, y, w, h, SEG_EPAISSEUR, SEG_MARGE)):
        if (code >> (6 - i)) & 1: rect_plein(oled, rx, ry, rw, rh, 1)
def dessiner_points(oled, x, y, w, h):
    cx = x + w // 2; y_centre = y + h // 2
    y1 = y_centre - ECART_VERTICAL_POINTS // 2
    y2 = y_centre + ECART_VERTICAL_POINTS // 2
    rect_plein(oled, cx - 1, y1 - 1, LARGEUR_POINTS, LARGEUR_POINTS, 1)
    rect_plein(oled, cx - 1, y2 - 1, LARGEUR_POINTS, LARGEUR_POINTS, 1)
def dessiner_chrono(oled, minutes_str, milli_str):
    x = X_BOITE0; y = Y_BOITE0; h = HAUTEUR_BOITE
    dessiner_digit(oled, x, y, LARGEUR_BOITE, h, minutes_str[0]); x += LARGEUR_BOITE + ECART_ENTRE_BOITES
    dessiner_digit(oled, x, y, LARGEUR_BOITE, h, minutes_str[1]); x += LARGEUR_BOITE + ECART_AVANT_POINTS
    LARGEUR_BOITE_POINTS = max(LARGEUR_POINTS + 2, 5)
    dessiner_points(oled, x, y, LARGEUR_BOITE_POINTS, h)
    x += LARGEUR_BOITE_POINTS + ECART_APRES_POINTS
    dessiner_digit(oled, x, y, LARGEUR_BOITE, h, milli_str[0]); x += LARGEUR_BOITE + ECART_ENTRE_BOITES
    w5 = LARGEUR_BOITE5; w_half = (w5 - ECART_INTERNE_DANS_BOITE5) // 2
    dessiner_digit(oled, x, y, w_half, h, milli_str[1])
    dessiner_digit(oled, x + w_half + ECART_INTERNE_DANS_BOITE5, y, w_half, h, milli_str[2])

# ========================= OLED / SPLASH (cœur 0) ===================
def init_oled():
    i2c = I2C(0, sda=Pin(I2C_SDA_PIN), scl=Pin(I2C_SCL_PIN), freq=I2C_FREQ)
    oled = SSD1306_I2C(OLED_W, OLED_H, i2c, addr=I2C_ADDR)
    oled.fill(0); oled.show()
    return oled
def splash_logo(oled):
    oled.fill(1); draw_logo(oled, x0=5, y0=2); oled.show()
    time.sleep_ms(SPLASH_DUREE_MS)
    oled.fill(0); oled.show()

# ========================= ETAT PARTAGÉ (mutex) =====================
# etat: 0=IDLE,1=RUN,2=STOPPED — accum_ms & start_ts gérés par cœur 1
ETAT_IDLE, ETAT_RUN, ETAT_STOPPED = 0, 1, 2
_state_lock = _thread.allocate_lock()
_state = {
    "etat": ETAT_IDLE,
    "accum_ms": 0,
    "start_ts": None,
    "beep_req": False,   # demandé par cœur 1, servi par cœur 0
}

def set_state(**kwargs):
    with _state_lock:
        _state.update(kwargs)

def get_state():
    with _state_lock:
        return _state["etat"], _state["accum_ms"], _state["start_ts"], _state["beep_req"]

# ========================= THREAD CHRONO (cœur 1) ===================
def thread_chrono():
    bp_am  = Pin(BP_AM_PIN,  Pin.IN, Pin.PULL_UP)
    bp_raz = Pin(BP_RAZ_PIN, Pin.IN, Pin.PULL_UP)

    last_am = bp_am.value(); last_raz = bp_raz.value()
    last_am_t = time.ticks_ms(); last_raz_t = time.ticks_ms()

    # Copie locale de l'état
    etat, accum_ms, start_ts, _ = get_state()

    while True:
        now = time.ticks_ms()

        # --- lecture boutons avec anti-rebond (polling)
        cur_am  = bp_am.value()
        cur_raz = bp_raz.value()

        # A/M : front descendant
        if last_am == 1 and cur_am == 0 and time.ticks_diff(now, last_am_t) >= DEBOUNCE_MS:
            last_am_t = now
            if etat == ETAT_RUN:
                # STOP → figer
                if start_ts is not None:
                    accum_ms += time.ticks_diff(now, start_ts)
                start_ts = None
                etat = ETAT_STOPPED
                set_state(etat=etat, accum_ms=accum_ms, start_ts=start_ts, beep_req=True)
            elif etat == ETAT_IDLE:
                # START depuis IDLE
                start_ts = now
                etat = ETAT_RUN
                set_state(etat=etat, accum_ms=accum_ms, start_ts=start_ts, beep_req=True)
            elif etat == ETAT_STOPPED:
                # Reprise interdite → rien
                pass

        # RAZ : front descendant — seulement si pas en RUN
        if last_raz == 1 and cur_raz == 0 and time.ticks_diff(now, last_raz_t) >= DEBOUNCE_MS:
            last_raz_t = now
            if etat != ETAT_RUN:
                accum_ms = 0
                start_ts = None
                etat = ETAT_IDLE
                set_state(etat=etat, accum_ms=accum_ms, start_ts=start_ts, beep_req=True)

        last_am = cur_am; last_raz = cur_raz

        # --- entretien RUN (pas de rendu ici) ---
        if etat == ETAT_RUN and start_ts is None:
            # Sécurité (ne devrait pas arriver)
            start_ts = now
            set_state(start_ts=start_ts)

        time.sleep_ms(3)

# ========================= MAIN (cœur 0) ============================
def main():
    oled = init_oled()
    adc_vsys = init_adc_vsys()
    adc_temp = init_adc_temp()

    # Splash
    splash_logo(oled)

    # ---------- MODE DIAG : RAZ maintenu au boot (comme v2.2.1) ----------
    bp_raz_boot = Pin(BP_RAZ_PIN, Pin.IN, Pin.PULL_UP)
    bp_am_boot  = Pin(BP_AM_PIN,  Pin.IN, Pin.PULL_UP)
    if bp_raz_boot.value() == 0:
        time.sleep_ms(50)
        if bp_raz_boot.value() == 0:
            while True:
                vsys = lire_vsys(adc_vsys)
                en_charge = charge_en_cours()
                temp_c = lire_temp_c(adc_temp)

                oled.fill(0)
                afficher_temperature_gauche(oled, temp_c)              # remplace CHRONO
                afficher_vsys_dans_batterie(oled, vsys, eclair=en_charge)
                oled.show()

                # sortie par A/M
                if bp_am_boot.value() == 0:
                    time.sleep_ms(20)
                    if bp_am_boot.value() == 0:
                        beep_now()
                        break
                time.sleep_ms(200)

    # ---------- Lancer le cœur 1 (logique chrono + boutons) ----------
    _thread.start_new_thread(thread_chrono, ())

    # ---------- Boucle affichage (cœur 0) ----------
    t_next_draw  = time.ticks_add(time.ticks_ms(), PERIODE_RAFRAICH_MS)
    t_next_batt  = time.ticks_add(time.ticks_ms(), PERIODE_BATT_MS)
    blink_on     = True
    t_next_blink = time.ticks_add(time.ticks_ms(), BLINK_PERIODE_MS)

    vsys = lire_vsys(adc_vsys)
    usb_present = (vsys >= USB_VSYS_MIN)
    en_charge = charge_en_cours()
    niveau = niveau_barres_pour_vsys(vsys)

    while True:
        now = time.ticks_ms()

        # service buzzer
        _service_beep(now)

        # Beep demandé par cœur 1 ?
        etat, accum_ms, start_ts, beep_req = get_state()
        if beep_req:
            beep_now(now)
            set_state(beep_req=False)

        # Mesures périodiques (batterie / charge)
        if time.ticks_diff(now, t_next_batt) >= 0:
            t_next_batt = time.ticks_add(t_next_batt, PERIODE_BATT_MS)
            vsys = lire_vsys(adc_vsys)
            usb_present = (vsys >= USB_VSYS_MIN)
            en_charge = charge_en_cours()
            niveau = niveau_barres_pour_vsys(vsys)

        # Clignotement batterie vide
        if time.ticks_diff(now, t_next_blink) >= 0:
            t_next_blink = time.ticks_add(t_next_blink, BLINK_PERIODE_MS)
            blink_on = not blink_on

        # Rendu périodique
        if time.ticks_diff(now, t_next_draw) >= 0:
            t_next_draw = time.ticks_add(t_next_draw, PERIODE_RAFRAICH_MS)

            # Calcul chrono (côté affichage pour une seule source de vérité)
            if etat == ETAT_RUN and start_ts is not None:
                elapsed = accum_ms + time.ticks_diff(now, start_ts)
            else:
                elapsed = accum_ms
            elapsed = max(0, elapsed) % 100000
            minutes = (elapsed // 1000) % 100
            milli   = elapsed % 1000
            min_str = f"{minutes:02d}"
            milli_str = f"{milli:03d}"

            batterie_critique = (niveau == 0) and (not usb_present) and (not en_charge)
            if batterie_critique and etat == ETAT_RUN:
                # Force arrêt propre côté état partagé
                if start_ts is not None:
                    accum_ms += time.ticks_diff(now, start_ts)
                set_state(etat=ETAT_STOPPED, accum_ms=accum_ms, start_ts=None)

            # DESSIN (identique v2.2.1)
            oled.fill(0)

            # Bandeau CHRONO + symbole
            x_fin_texte = dessiner_texte_5x7(oled, CHRONO_X, CHRONO_Y, TEXTE_CHRONO, 1, 1, 1)
            x_sym = x_fin_texte + SYMBOLE_ESPACE_APRES_TEXTE
            y_sym = CHRONO_Y
            if etat == ETAT_RUN:
                dessiner_triangle_7(oled, x_sym, y_sym, 1)
            elif etat == ETAT_STOPPED:
                dessiner_carre_7(oled, x_sym, y_sym, 1)

            if batterie_critique:
                afficher_message_recharger(oled)
                if blink_on:
                    dessiner_batterie(oled, X_BATT, Y_BATT, 0, eclair=en_charge)
            else:
                dessiner_chrono(oled, min_str, milli_str)
                dessiner_batterie(oled, X_BATT, Y_BATT, niveau, eclair=en_charge)

            oled.show()

        time.sleep_ms(3)

# ------------------------- LANCEMENT --------------------------------
if __name__ == "__main__":
    main()
